﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.Tilemaps;

//PathFinter is a static class that implements the A* algorithm to find traversable paths between points in the city
public static class Pathfinder
{
    /// <summary>
    /// Attempts to find a logically traversable path between 2 points in the city
    /// </summary>
    /// <param name="startPos">The starting position of the path</param>
    /// <param name="endPos">The end position of a path</param>
    /// <returns>A list of all nodes in the path between the positions, or null if failed</returns>
    public static List<PathNode> FindPath(Vector3Int startPos, Vector3Int endPos)
    {
        //Create our nodes and lists
        PathNode startNode = new PathNode(startPos);
        PathNode endNode = new PathNode(endPos);
        List<PathNode> openList = new List<PathNode>() { startNode };
        HashSet<PathNode> closedList = new HashSet<PathNode>();

        //Create our starting node
        startNode.GCost = CalculateDistanceCost(startNode, endNode);
        startNode.HCost = CalculateDistanceCost(startNode, endNode);
        startNode.CalculateFCost();

        while(openList.Count > 0)
        {
            if(openList.Count >= 50 || closedList.Count >= 10000)    //Safety check, let's not get stuck in a huge loop
            {
                Debug.LogWarning(string.Format("WARNING: Broke from A* FindPath due to excessively large lists. Start Node Position: {0}, End Node Position: {1}, Is there an implementation bug?", startNode.Position, endNode.Position));
                break;
            }

            PathNode currentNode = GetLowestFCostNode(openList);    //Get the least costly node

            if(currentNode.Position == endNode.Position)
            {
                //We've made it, calculate the path and return it
                return CalculatePath(currentNode);
            }

            else
            {
                //We've checked this node, let's remove it from the open list and add it to the closed list
                openList.Remove(currentNode);
                closedList.Add(currentNode);

                //Get and loop through all neighbours
                List<PathNode> neighbouringNodes = GetNeighbouringNodes(currentNode);
                foreach (PathNode neighbourNode in neighbouringNodes)
                {
                    if (closedList.Any(x => x.Position == neighbourNode.Position))
                    {
                        //We've already checked this neighbour
                        continue;
                    }

                    else
                    {
                        int thisGCost = currentNode.GCost + CalculateDistanceCost(currentNode, neighbourNode);  //Calculate the cost to this neighbour from where we are

                        if(thisGCost < neighbourNode.GCost)
                        {
                            //It's cheaper, set up the neighbour's new values
                            neighbourNode.Parent = currentNode;
                            neighbourNode.GCost = thisGCost;
                            neighbourNode.HCost = CalculateDistanceCost(neighbourNode, endNode);
                            neighbourNode.CalculateFCost();

                            //Now add it to the open list
                            int openIndex = openList.FindIndex(x => x.Position == neighbourNode.Position);
                            if (openIndex == -1)
                            {
                                openList.Add(neighbourNode);
                            }

                            else
                            {
                                openList[openIndex] = neighbourNode;
                            }
                        }
                    }
                }
            }
        }

        return null;    //Didn't find a path
    }

    /// <summary>
    /// Performs iteration to compute a list of nodes from the endNode back to its root parent
    /// </summary>
    /// <param name="endNode">The node from which to compute a path back from</param>
    /// <returns>A list of nodes in the path</returns>
    private static List<PathNode> CalculatePath(PathNode endNode)
    {
        List<PathNode> retList = new List<PathNode>() { endNode };
        PathNode currentNode = endNode;

        //Keep looping through, add all the subsequent parents
        while(currentNode.Parent != null)
        {
            retList.Add(currentNode.Parent);
            currentNode = currentNode.Parent;
        }

        retList.Reverse();  //Reverse the list so we start from the root parent
        return retList;
    }
    
    /// <summary>
    /// Returns a list of nodes that neighbour the parent node
    /// </summary>
    /// <param name="parentNode">The parent node to check neighbours for</param>
    /// <returns>A list of neighbouring nodes</returns>
    private static List<PathNode> GetNeighbouringNodes(PathNode parentNode)
    {
        Vector3Int leftNeighbourPos = new Vector3Int(parentNode.Position.x - 1, parentNode.Position.y, parentNode.Position.z);
        Vector3Int rightNeighbourPos = new Vector3Int(parentNode.Position.x + 1, parentNode.Position.y, parentNode.Position.z);
        Vector3Int topNeighbourPos = new Vector3Int(parentNode.Position.x, parentNode.Position.y + 1, parentNode.Position.z);
        Vector3Int bottomNeighbourPos = new Vector3Int(parentNode.Position.x, parentNode.Position.y - 1, parentNode.Position.z);

        List<Vector3Int> possibleNeighbourPositions = new List<Vector3Int>() { leftNeighbourPos, rightNeighbourPos, topNeighbourPos, bottomNeighbourPos };
        List<PathNode> retNeighbours = new List<PathNode>();

        foreach(Vector3Int neighbourPos in possibleNeighbourPositions)
        {
            if(GameController.Instance.LSystem.Drawer.RoadPositions.Contains(neighbourPos))
            {
                retNeighbours.Add(new PathNode(neighbourPos));
            }
        }

        return retNeighbours;
    }

    /// <summary>
    /// Computes a heuristic distance cost between 2 nodes
    /// </summary>
    /// <param name="startNode">The node to start at</param>
    /// <param name="endNode">The ending node</param>
    /// <returns>The cost between the two nodes</returns>
    private static int CalculateDistanceCost(PathNode startNode, PathNode endNode)
    {
        //Really simple but effective distance cost heuristic, just compute the magnitude of the vector between the nodes' positions
        return Convert.ToInt32((startNode.Position - endNode.Position).magnitude * 100.0f);
    }

    /// <summary>
    /// Gets the nodes with the lowest F cost from the list
    /// </summary>
    /// <param name="nodes">A list of nodes to check</param>
    /// <returns>The node with the lowest F cost</returns>
    private static PathNode GetLowestFCostNode(List<PathNode> nodes)
    {
        PathNode lowestFCostNode = nodes[0];

        for(int i = 1; i < nodes.Count; i++)
        {
            if(nodes[i].FCost < lowestFCostNode.FCost)
            {
                lowestFCostNode = nodes[i];
            }
        }

        return lowestFCostNode;
    }
}
